function op = opWavelet_AG(m, n, family, filter, levels, sol_type, wavelet_type)
% OPWAVELET_AG  Wavelet operator
%
%    OPWAVELET_AG(M,N,FAMILY,FILTER,LEVELS,SOL_TYPE, WAVELET_TYPE)
%    creates a wavelet operator of given FAMILY, for M by N matrices. The
%    wavelet transformation is computed using the Rice Wavelet Toolbox.
%
%    The remaining three parameters are optional. FILTER = 8
%    specifies the filter length and must be even. LEVELS = 5 gives
%    the number of levels in the transformation. Both M and N must
%    be divisible by 2^LEVELS. SOL_TYPE = 'min' indicates what type of
%    solution is desired; 'min' for minimum phase, 'max' for
%    maximum phase, and 'mid' for mid-phase solutions.  WAVELET_TYPE = 1
%    indicates the use of the DWT and = 2 indicates the use of the UDWT

%   Copyright 2008, Rayan Saab, Ewout van den Berg and Michael P. Friedlander
%   http://www.cs.ubc.ca/labs/scl/sparco
%   $Id: opWavelet.m 1040 2008-06-26 20:29:02Z ewout78 $
%
% Modified on October 2012 by Adriana Gonzalez
% Last Update on May 2013 by Adriana Gonzalez

if (nargin < 3), family = 'Daubechies'; end;
if (nargin < 4), filter =     8;        end;
if (nargin < 5), levels =     5;        end;
if (nargin < 6), sol_type   = 'min';    end;
if (nargin < 7), wavelet_type   = 1;    end;

family = lower(family);

switch family
    
    case {'daubechies'}
        h = daubcqf(filter);
        
    case {'haar'}
        h = daubcqf(0);
        
    otherwise
        error('Wavelet family %s is unknown', family);

end
    
if(wavelet_type == 1)
    op = @(x,mode) opWavelet_AG_intrnl(m,n,family,filter,levels,sol_type,h,x,mode);
else
    op = @(x,mode) opWavelet_AG_UDWT_intrnl(m,n,family,filter,levels,sol_type,h,x,mode);
end

function y = opWavelet_AG_intrnl(m,n,family,filter,levels,type,h,x,mode)
checkDimensions(m*n,m*n,x,mode);
if mode == 0
   y = {m*n,m*n,[0,1,0,1],{'Wavelet DWT',family,filter,levels,type}};
elseif mode == 1
    if isreal(x)
        y = midwt(reshape(x,[m,n]), h, levels);
    else
        y1 = midwt(reshape(real(x),[m,n]), h, levels);
        y2 = midwt(reshape(imag(x),[m,n]), h, levels);
        y = y1 + sqrt(-1) * y2;
    end
    y = reshape(y,[m*n,1]);  
else    
    if isreal(x)
        y = mdwt(reshape(x,[m,n]), h, levels);
    else
        y1 = mdwt(reshape(real(x),[m,n]), h, levels);
        y2 = mdwt(reshape(imag(x),[m,n]), h, levels);
        y = y1 + sqrt(-1) * y2;
    end
    y = reshape(y,[m*n,1]);
end

function y = opWavelet_AG_UDWT_intrnl(m,n,family,filter,levels,type,h,x,mode)
checkDimensions(m*n,m*n*(1+3*(levels-1)),x,mode);
if mode == 0
    y = {m*n,m*n*(1+3*(levels-1)),[0,1,0,1],...
        {'Wavelet UDWT',family,filter,levels,type}};
elseif mode == 1
    y = mirdwt_TI2D(reshape(x,[m,n*(1+3*(levels-1))]), h, levels);
    y = y(:);
else
    y = mrdwt_TI2D(reshape(x,[m,n]), h, levels);
    y = y(:);
end
