function [x,iter,T,SNR,p,d,condition,time,l1norm_x,l1norm_gt] = ...
    cvx_unwrapping(qy,epsilon1,epsilon2,w,gt,x0,param)

% [x,iter,T,SNR,p,d,condition,time,l1norm_x,l1norm_gt] = ...
%     cvx_unwrapping(q,epsilon1,epsilon2,w,gt,x0,param);
%
% Usage:
% This algorithm solves the following minimization problem [1]:
%
% [x*,v*] = argmin_[u,v] || W^T u ||_1
%                   s.t. || v || <= epsilon1
%                        || q - grad(u+v) ||_1 <= epsilon2
%                        u(1) = 0
%
% using the algorithm of Chambolle-Pock [2,3].
%
% Inputs
%
%   * q: Vector of size Nx1. Function of the wrapped phase (indirect
%   observations)
%   * epsilon1: scalar. Bound on the norm of || v ||_2.
%   * epsilon2: scalar. Bound on the norm of || q - grad(u+v) ||_1.
%   * w: matlab structure decribing the wavelet basis. (Described below)
%   * gt: Ground Truth vector to compute the SNR at each iteration
%   * x0: vector. Initialization for the CP algorithm. (Default: zeros).
%   * param: matlab structure. Parameters needed for the CP algorithm.
%   (Described below) 
%
% The matlab structure w contains the follwing fields:
%   * w.name: string. Specifies name of the wavelet (Default = 'daubechies')
%   * w.filter: scalar. Defines the length of the filter. (Default = 14)
%   * w.level: scalar. Defines the amount of levels in the wavelet
%   decomposition. (Default = 7)
%   * w.type: scalar. (Default = 1)
%
% The matlab structure param contains the follwing fields:
%  * param.max_iter: Maximum number of iterations when the number of
%  iterations is not especified (Default 1e4)
%  * param.Th: scalar. Stop criterion. (Default = [1e-4 5e-1 5e-1])
%  * param.iter: scalar. Number of iterations. If not specified, the max
%  number of iterations will be max_iter (Default = [])
%  * param.ThMode: 1 => Th related to stability
%                  2 => Th related to primal and dual residuals
%                  3 => Th related to both stability and residuals
%                       (Default)
%  * param.res_norm: how to compute the residual norms (Default = 2)
%  * param.t: scalar. To set the relation between the CP parameters 
%       => sigma = param.t*tau (Default = 1e3)
%  * param.print: boolean. If true it prints the values at each iteration
%
% Outputs:
%
%  * x: vector of size Nx1. Estimated signal.
%  * iter: Scalar. Total number of iterations the algorithm had to do to
%    compute until the stop criterion was accomplished.
%  * T: array. Convergence criterion norm(x_est{i+1} -
%  x_est{i})/norm(x_est{i}) 
%  * SNR: 1x(iter/10) array with the SNR value at each 10 iterations
%  * p: 1xiter array. norm of the primal residual.
%  * d: 1xiter array. norm of the dual residual.
%  * condition: 1x(iter/10) array. Residual convergence of the algorithm
%  * l1norm_x: L1 norm of the estimated variable
%  * l1norm_gt: L1 norm of the GT
%
% References:
%
% [1] A. Gonzalez and L. Jacques. ''Robust Phase Unwrapping by Convex 
% Optimization,'' in IEEE International Conference on Image Processing
% (ICIP14), Paris, France. October 27-30, 2014.
% 
% [2] Chambolle, A. & Pock, T. ''A first-order primal-dual algorithm for
% convex problems with applications to imaging,'' Technical Report, CMAP,
% Ecole Polytechnique. May 2010.
%
% [3] Gonzalez, A. Jacques, L., De Vleeschouwer, C. and Antoine, P.
% ''Compressive Optical Deflectometric Tomography: A constrained
% Total-Variation Minimization Approach,'' Inverse Problems and Imaging, 
% Vol. 8, no. 2, p.~421-457 (2014). arXiV:1209.0654 
%
% [4] SPARCO Toolbox. E. v. Berg, M. P. Friedlander, G. Hennenfent, F.
% Herrmann, R. Saab, and O. Yilmaz, ''Sparco: A testing framework for
% sparse reconstruction,'' Dept. Computer Science, University of British
% Columbia, Vancouver, Tech. Rep. TR-2007-20, October 2007.
%
% [5] Rice Wavelet Toolbox. 
% http://dsp.rice.edu/software/rice-wavelet-toolbox
%
% BibTeX Citation when using this code:
% @INPROCEEDINGS{7025343, 
% author={Gonzalez, A. and Jacques, L.}, 
% booktitle={Image Processing (ICIP), 2014 IEEE International Conference on}, 
% title={Robust phase unwrapping by convex optimization}, 
% year={2014}, 
% pages={1713-1717},
% doi={10.1109/ICIP.2014.7025343}, 
% month={Oct},}
%
% Copyright 2014 Adriana Gonzalez 
% PhD Student at Universite catholique de Louvain (UCL), Belgium

% The Rice Wavelet Toolbox (RWT) must be installed and must be located in
% the following path 
addpath(genpath(strcat(pwd,'\rwt')))

% *************************************************************************
%                             INITIALIZATION
% *************************************************************************

if ~isfield(w,'name'); w.name = 'daubechies'; end
if ~isfield(w,'filter'); w.filter = 14; end
if ~isfield(w,'level'); w.level = 7; end
if ~isfield(w,'type'); w.type = 1; end

if ~isfield(param,'iter'); param.iter = []; end
if ~isfield(param,'ThMode'); param.ThMode = 3; end
if ~isfield(param,'Th'); param.Th = [1e-4 5e-1 5e-1]; end
if ~isfield(param,'max_iter'); param.max_iter = 1e4; end
if ~isfield(param,'res_norm'); param.res_norm = 2; end
if ~isfield(param,'t'); param.t = 1e3; end
if ~isfield(param,'print'); param.print = 0; end

N = length(qy)/2;
n = sqrt(N);

if isempty(x0); x0 = zeros(2*N,1); end

% *************************************************************************
%                             WAVELET OPERATOR
% *************************************************************************

opW = opTranspose(opWavelet_AG(n,n,w.name,w.filter,w.level,'min',w.type));

Wmask = 0*opW(qy(1:N),1); NW = length(Wmask(:));

% Set Theta containing the detail coefficients
L = w.level;
if w.type==1
    Theta = zeros(N,1);
    for ii=1:sqrt(N)/(2^L)
        Theta((ii-1)*n+1:(ii-1)*n+(n/(2^L))) = 1;
    end
else
    Theta = zeros(NW,1);
    Theta(1:N) = 1;
end
Theta = find(Theta==0);
opS_Theta = opRestriction(NW,Theta);

% *************************************************************************
%                             GRADIENT OPERATOR
% *************************************************************************

opG = opGrad_AG(n);

% *************************************************************************
%                            SELECTION OPERATORS
% *************************************************************************

opSu = opRestriction(2*N,1:N);
opSv = opRestriction(2*N,N+1:2*N);

% *************************************************************************
%                                SUM OPERATOR
% *************************************************************************

opZ = opSum([1 1],opSu,opSv);

% *************************************************************************
%                                OPERATORS CP
% *************************************************************************

op1 = opFoG(opS_Theta,opW,opSu); % First applies opSu, then opW and then opS_Theta
op2 = opSv;
op3 = opFoG(opG,opZ); % First applies opZ and then opG

% Concatenation of all operators
op = {op1; op2; op3};

% Operator norm
opCP = opBlockDiag(op1,op2,op3);
L_CP = OpNorm(opCP,rand(6*N,1),100,1e-5);
if param.print;
    fprintf('GCP-> ||A||_2 = %3.5f\n', L_CP);
end

% *************************************************************************
%                           PROXIMAL OPERATORS
% *************************************************************************

% Proximity operator of the l1 norm
proxF1 = @(z, sigma) SoftTh(z,sigma);
% Proximity operator of the dual of the l1 norm
proxF1d = @(z, sigma) z - sigma*proxF1(z*(1/sigma),(1/sigma));

% Proximity operator of the indicator function in convex set
% C1:{ v in R^{Nx1} : || v ||_2 <= epsilon1 }
proxF2 = @(z, sigma) z*min((epsilon1/norm(z)),1);
% Proximity operator of the dual indicator function
proxF2d = @(z, sigma) z - sigma*proxF2(z*(1/sigma),(1/sigma));

% Proximity operator of the indicator function in convex set
% C:{ v in R^{2Nx1} : || qy - v ||_1 <= epsilon2 }
proxF3 = @(z, sigma) qy + Proj_l1ball(z-qy,epsilon2); 
% Proximity operator of the dual indicator function
proxF3d = @(z, sigma) z - sigma*proxF3(z*(1/sigma),(1/sigma));

% Concatenation of all proximal operators related to the dual of F
proxFd = {proxF1d; proxF2d; proxF3d};

% Proximity operator of zero function
proxH = @(z,tau) fixedvalueregions(z,1,0);

% *************************************************************************
%                          OBJECTIVE FUNCTION
% *************************************************************************
    
f = @(x) norm(op1(x(:),1),1);
    
% *************************************************************************
%                           QUALITY FUNCTION
% *************************************************************************

gt_aux = gt(1:N) - mean(gt(1:N));
        
q.name = 'RSNR';
q.function = ...
    @(x) 20*log10(norm(gt_aux(:))/norm((x(1:N)-mean(x(1:N)))-gt_aux(:)));
    
% *************************************************************************
%                           ALGORITHM
% *************************************************************************
   
param.tau = sqrt(0.9/param.t)/L_CP;
param.sigma = param.t*param.tau;
if param.print
    fprintf('tau = %d and sigma = %d \n',param.tau,param.sigma);
end

tStart = tic;
[x_est,iter,T,p,d,condition,l1norm_x,SNR] = ...
    GCP_nonadaptive_NF(op,proxFd,proxH,param,x0,q,f);
time = toc(tStart);

x = x_est{2}; 

l1norm_gt = norm(op1(gt,1),1);
fprintf('\n|| W^T x ||_1 = %e\n',l1norm_gt);

l1_x = norm(op1(x,1),1);
fprintf('|| W^T x* ||_1 = %e\n',l1_x);

fprintf('\nepsilon = %e\n',epsilon1);
l2_v = norm(x(N+1:end),2);
fprintf('|| v* ||_2 = %e\n',l2_v);

l2_diff = norm(qy-op3(x,1),1);
fprintf('\n|| q - grad(x*+v*) ||_1 = %e\n',l2_diff);
