function [x_est,iter,T,p,d,condition,f_eval,q_eval] = ...
    GCP_nonadaptive_NF(op,proxFd,proxH,param,x0,q,f)

% [x_est,iter,T,SNR,p,d,condition,f_eval] = ...
%    GCP_nonadaptive_NF_v2(op,proxFd,proxH,param,x0,q,f)
%
% Usage:
% This algorithm solves the following minimization problem
%
%       t* = argmin_t sum_i^N F(K_i u_i) + H(u_1);  
%       with t = (u_1,...,u_N) and u_1=...=u_N
%
%       t* = argmin_t F(Kt) + G(t); with K = diag(K_i)
%
% Inputs:
%  * op: cell of N operators with input (x,mode). Operator K
%  * proxFd: cell of N operators with input (z,gamma). Proximity operator 
%  of (F_i)dual. Defined for vectors
%  * proxH: operator with input (z,gamma). Proximity operator of H. 
%  Defined for vectors
%  * param: matlab structure. Parameters needed for the algorithm, 
%  as described below
%  * x0: initialization vector for variable x_est
%  * q: matlab structure. 
%      - q.name: name of the quality function
%      - q. function: function handle containing the quality function
%  * f: function handle containing the objective function
%
% The matlab structure param contains the follwing fields:
%  * param.sigma: scalar. Initialization of the parameter associated with proxFd.
%  * param.tau: scalar. Initialization of the parameter associated with proxG.
%  * param.max_iter: Maximum number of iterations when the number of iterations
%    is not especified. Default = 500
%  * param.Th: vector 1x3. Stop criterion. Default = [1e-4 0.5 0.5].
%  * param.iter: scalar. Number of iterations. Default = param.max_iter.
%  * param.ThMode: 1 => Th related to stability. (Default)
%                  2 => Th related to primal and dual residuals.
%                  3 => Th related to both stability and residuals.
%  * param.res_norm: norm used for the residual norms: l1,l2. Default = 1
%  * param.print: boolean. If true it prints the values at each iteration
% 
% Outputs:
%  * x_est: 2x1 cell. Estimated signal. The first cell contains the
%    estimated signal in the previous iteration and the second one
%    contains the value of the estimated signal in the last iteration. Both
%    cells contain (n^2)x1 vectors.
%  * iter: Scalar. Total number of iterations the algorithm had to
%    compute until the stop criterion was accomplished.
%  * T: array. Convergence criterion ||x^{k+1} - x^{k}||_2 / ||x^{k}||_2
%  * p: 1xiter array. l1-norm of the primal residual.
%  * d: 1xiter array. l1-norm of the dual residual.
%  * condition: 1x(iter/10) array. Residual convergence of the algorithm
%    ||p||_2^2 + ||d||_2^2
%  * f_eval: 1x(iter/10) array. Objective function evaluated each 10 iter
%  * q_eval: 1x(iter/10) array with the value of the quality function q at
%    each 10 iterations 
%
% Comments:
%  Proximity operators are defined as follows: 
%       prox_{gamma f} z = argmin_y gamma*f(y) + (1/2)*||z - y||^2
%
% References:
% 
% [1] Chambolle, A. & Pock, T. "A first-order primal-dual algorithm for convex
% problems with applications to imaging". Technical Report, CMAP, École
% Polytechnique. May 2010.
%
% [2] Gonzalez, A. Jacques, L., De Vleeschouwer, C. and Antoine, P.
% ''Compressive Optical Deflectometric Tomography: A constrained
% Total-Variation Minimization Approach,'' Inverse Problems and Imaging, 
% Vol. 8, no. 2, p.~421-457 (2014). arXiV:1209.0654 
%
% BibTeX Citation when using this code:
% @article{GJDA2014,
% 	author = {Adriana  Gonz\'{a}lez and Laurent  Jacques and Christophe De  Vleeschouwer and Philippe  Antoine},
% 	title = {Compressive optical deflectometric tomography: A constrained total-variation minimization approach},
% 	journal = {Inverse Problems and Imaging},
% 	volume = {8},
% 	number = {2},
% 	pages = {421-457},
% 	year ={2014},
% 	issn = {1930-8337},
% 	doi = {10.3934/ipi.2014.8.421},
% 	url = {http://aimsciences.org/journals/displayArticlesnew.jsp?paperID=9905},}
%
% Copyright 2014 Adriana Gonzalez 
% PhD Student at Universite catholique de Louvain (UCL), Belgium

% Initialization

infit = false; 

% Global parameters
if ~isfield(param, 'max_iter'); param.max_iter = 500; end
if ~isfield(param, 'Th'); param.Th = [1e-4 5e-1 5e-1]; end
if (~isfield(param, 'iter') || isempty(param.iter)); 
    param.iter = param.max_iter; infit = true;
end
if ~isfield(param,'ThMode'); param.ThMode = 1; end
if ~isfield(param,'res_norm'); param.res_norm = 1; end
if ~isfield(param,'print'); param.print = 'true'; end

if (isempty(q)); q.name = 'none'; q.function = @(x) x; end;
if (isempty(f)); f = @(x) x; end; 

% Amount of operators
NOp = length(op);

% Initialization dual variable with zeros and size of dual variable
u = cell(NOp,2);
NDual = zeros(NOp,1);
NDual_Tot = 0;
for ii=1:NOp
    info = op{ii}([],0); 
    NDual(ii) = info{1};
    u{ii,2} = zeros(NDual(ii),1);
    NDual_Tot = NDual_Tot + NDual(ii);
end

% Initialization primal variable with x0 or with zeros if x0 is empty
x_est = cell(2,1);
if (isempty(x0))
    x_est{2} = zeros(info{2},1);
else
    x_est{2} = x0;
end

% Initialization auxiliar variable with zeros
v = cell(2,1);
v{2} = zeros(info{2},1);

% Values to save each iteration
p = zeros(1,param.iter);
d = zeros(1,param.iter);
condition = zeros(1,param.iter);

% Values to save each 10 iterations
q_eval = zeros(1,round(param.iter/10));
f_eval = zeros(1,round(param.iter/10));
T = ones(1,round(param.iter/10));

% Algorithm

j = 1;

iter = param.iter;

for k = 1 : iter
    
    % Initialization
    x_est{1} = x_est{2};
    v{1} = v{2};
    
    % Dual Optimization - 1st step
    z = zeros(info{2},1);
    for ii=1:NOp
        u{ii,1} = u{ii,2};
        u{ii,2} = proxFd{ii}((u{ii,1} + param.sigma * op{ii}(v{1},1)),param.sigma);
        z = z + op{ii}(u{ii,2},2);
    end
    
    % Primal Optimization - 2nd step        
    x_est{2} = proxH(x_est{1} - 1/NOp * param.tau * z,(param.tau/NOp));
        
    % Updating primal variable - 3rd step
    v{2} = 2*x_est{2} - x_est{1}; 
    
    % Primal Residual
    primal = (NOp/param.tau)*(x_est{1} - x_est{2});
    p(k) = norm(primal,param.res_norm);
    
    % Dual Residual
    dual = zeros(NDual_Tot,1); 
    jj = 0;
    for ii=1:NOp
        dual(jj+1:jj+NDual(ii)) = ...
            (1/param.sigma)*(u{ii,1} - u{ii,2}) - op{ii}(x_est{2}-v{1},1);
        jj = jj + NDual(ii);
    end        
    d(k) = norm(dual,param.res_norm);      
           
    % Computing Convergence
    if (mod(k,10) == 0) && (norm(x_est{1,1}) ~= 0)  
        
        % ||p||_2^2 + ||d||_2^2
        condition(j) = (norm(primal,2))^2 + (norm(dual,2))^2;
        % Th = ||x^{k+1}-x^{k}||_2 / ||x^{k}||_2
        T(j) = norm(x_est{2} - x_est{1})/norm(x_est{1});
        % Evaluation of the objective function
        f_eval(j) = f(x_est{2});
        % Evaluation of the quality function
        q_eval(j) = q.function(x_est{2});
        if param.print
            fprintf('\nobj_f = %e\n%s = %ddB\nTh = %e\n%d iter\n||p||_%d = %d\n||d||_%d = %d\n||p||_2^2 + ||d||_2^2 = %d\n'...
                ,f_eval(j),q.name,q_eval(j),T(j),k,param.res_norm,p(k),param.res_norm,d(k),condition(j));
        end
        
        % Stop Criterion: Th stability
        if ((T(j) < param.Th(1)) && infit && (param.ThMode==1))
            iter = k;
            break
        end
        
        % Stop Criterion: Th residuals
        if ((p(k) < param.Th(2)) && (d(k) < param.Th(3)) && infit && (param.ThMode==2))
            iter = k;
            break
        end
        
        % Stop Criterion: Th (either residuals or stability)
        if (((T(j) < param.Th(1)) || ((p(k) < param.Th(2)) && (d(k) < param.Th(3)))) && infit && (param.ThMode==3))
            iter = k;
            break
        end
                
        j = j + 1;
        
    end
        
end

T = T(1:round(iter/10));
p = p(:,1:iter);
d = d(:,1:iter);
condition = condition(:,1:round(iter/10));
f_eval = f_eval(1:round(iter/10));
q_eval = q_eval(1:round(iter/10));