function [L,e] = OpNorm(op,val,max_it,Th)

% [L,e] = OpNorm(op,val,max_it,Th)
%
% OpNorm computes the norm of operator 'op' and returns its value in 'L'.
% The algorithm is based on the eigenvalue standard power method.
%
% Inputs:
%  * op: Operator with input (x,mode), where x is the input column vector
%  (Nx1) and mode is the operator mode: 0-Information as a cell containing
%  {output vector length, input vector length}, 1-Forward, 2-Adjoint).
%  * val: Column vector (Nx1). Values for the initialization of the power
%  method. Default: Random values.
%  * max_it: Scalar. Maximum number of allowed iterations. Default: 100.
%  * Th: Scalar. Threshold value, algorithm stops when (s_{n+1}-s_{n})<Th.
%  Default: 1e-5.
%
% Outputs:
%  * L: Scalar. Indicates the operator (op) norm.
%  * e: Vector (max_it x 1). Value of L through the iterations.
% 
% References:
%
% [1] E.Sidky, J. Jorgensen & X. Pan. Convex optimization problem
% prototyping for image reconstruction in computed tomography with the
% Chambolle-Pock algorithm. 08 March 2012. Available in  arXiv:1111.5632v2.
%
% [2] A. Gonzalez, J. Jacques, C. De Vleeschouwer, and P. Antoine.
% Compressive Optical Deflectometric Tomography: A constrained
% Total-Variation Minimization Approach. Inverse Problems and Imaging
% Journal 8(2), pp. 421-457, 2014. Available in arXiV:1209.0654.
%
% Copyright 2014 Adriana Gonzalez 
% PhD Student at Universite catholique de Louvain (UCL), Belgium

% Operator Information
info = op([],0);
N = info{2};

% Default values if not specified
if (isempty(val)); val = rand(N,1); end 
if (isempty(max_it)); max_it = 100; end
if (isempty(Th)); Th = 1e-5; end

% Initialization
x = cell(2,1); x{2} = val;
s = zeros(2,1);
e = zeros(max_it,1);

% Algorithm

for n = 1:max_it
    
    s(1) = s(2);
    x{1} = x{2};
    x{2} = op(op(x{1},1),2);
    x{2} = x{2}/norm(x{2});
    s(2) = norm(op(x{2},1));
    e(n) = s(2);
    
    if (abs(s(2) - s(1)) <= Th) 
        break % Stop if algorithm is stable
    end
       
end

L = s(2);
